/*************************************************************************
 * The contents of this file are subject to the MYRICOM MYRINET          *
 * EXPRESS (MX) NETWORKING SOFTWARE AND DOCUMENTATION LICENSE (the       *
 * "License"); User may not use this file except in compliance with the  *
 * License.  The full text of the License can found in LICENSE.TXT       *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2004 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

#ifndef _mx_arch_klib_h
#define _mx_arch_klib_h

#include "mx_arch.h"

#include <sys/kthread.h>
#include <sys/sema.h>
#include <sys/sf_buf.h>
#include <sys/sx.h>
#include <sys/time.h>

/* kernel library symbols */

#define mx_klib_symbol(s)

/* kernel library wrappers */

#define mx_klib_sprintf(args...) sprintf(args)
#define mx_klib_printf(args...) printf(args)
#define mx_klib_snprintf(args...) snprintf(args)
#define mx_klib_strlen(s) strlen(s)

static inline int
mx_klib_gettimeofday(struct timeval *tv, struct timezone *tz) {
  if (tv != NULL)
    microtime(tv);
  return 0;
}

/* klib mutex:

   We use an SX (shared/exclusive) lock rather than a real mutex so as
   to not confuse the FreeBSD locking system when we sleep holding the
   klib mutex 
*/

typedef struct sx mx_klib_mutex_t;

#define mx_klib_mutex_init(mutexp) sx_init(mutexp, "mx klib sx lock")
#define mx_klib_mutex_destroy(mutexp) sx_destroy(mutexp)
#define mx_klib_mutex_lock(mutexp) sx_xlock(mutexp)
#define mx_klib_mutex_unlock(mutexp) sx_xunlock(mutexp)

/* klib thread */
typedef int mx_klib_thread_t;
typedef void mx_klib_thread_return_t;
static inline int
mx_klib_thread_create(mx_klib_thread_t *pid,
                      void (*func)(void *), void *arg) 
{
  struct proc *p;
  int ret = kthread_create(func, arg, &p, 0, 0, "mx_klib");
  if (!ret)
    *pid = p->p_pid;
  return ret;
}
static inline void 
mx_klib_thread_join(mx_klib_thread_t *pid) 
{
  struct proc *p;
  while (1) {
    p = pfind(*pid);
    if (p == 0)
      break;
    PROC_UNLOCK(p);
    tsleep(p, 0, "mxkthrjoin", hz);
  }
}
#define mx_klib_thread_exit(threadp) kthread_exit(0)
#define mx_klib_thread_init(threadp, name) /* nothing */

static inline int
mx_arch_memcpy_from_segment(void *void_ptr, uint64_t segment_ptr,
			    size_t amount, uintptr_t memory_context)
{
  int status = 0;
  switch (MX_PIN_CTX_TO_LOWLEVEL(memory_context)) {
  case MX_PIN_PHYSICAL:
    {
      vm_page_t page;
      struct sf_buf *sf;
      vm_offset_t offset = segment_ptr & (PAGE_SIZE-1);
      vm_paddr_t phys = segment_ptr;
      vm_offset_t remaining = amount;
      vm_offset_t to_copy;
      char *vaddr;
      char *ptr;

      /* hack around  GCC "pointer of type `void *' used in arithmetic" */
      ptr = (char *)void_ptr;  
      while (remaining) {
	to_copy = PAGE_SIZE - offset;
	if (to_copy > remaining)
	  to_copy = remaining;

	/* allocate kernel virtual address and map the page there */
	page = PHYS_TO_VM_PAGE(phys);
	sf = sf_buf_alloc(page, 0);
	if (!sf) {
	  MX_WARN(("mx_arch_memcpy_from_segment: sf_buf_alloc() failed !\n"));
	  status = ENOMEM;
	  goto exit;
	}
	vaddr = (char *)sf_buf_kva(sf);
	bcopy(vaddr+offset, ptr, to_copy);
	sf_buf_free(sf);
	remaining -= to_copy;
	ptr += to_copy;
	phys += to_copy;
	offset = 0;
      }
    }
    break;
  default:
    status = EINVAL;
    break;
  }
 exit:
  return status;
}

static inline int
mx_arch_memcpy_to_segment(uint64_t segment_ptr, void *void_ptr,
			  size_t amount, uintptr_t memory_context)
{
  int status = 0;
  switch (MX_PIN_CTX_TO_LOWLEVEL(memory_context)) {
  case MX_PIN_PHYSICAL:
    {
      vm_page_t page;
      struct sf_buf *sf;
      vm_offset_t offset = segment_ptr & (PAGE_SIZE-1);
      vm_paddr_t phys = segment_ptr;
      vm_offset_t remaining = amount;
      vm_offset_t to_copy;
      char *vaddr;
      char *ptr;

      /* hack around  GCC "pointer of type `void *' used in arithmetic" */
      ptr = (char *)void_ptr;  
      while (remaining) {
	to_copy = PAGE_SIZE - offset;
	if (to_copy > remaining)
	  to_copy = remaining;

	/* allocate kernel virtual address and map the page there */
	page = PHYS_TO_VM_PAGE(phys);
	sf = sf_buf_alloc(page, 0);
	if (!sf) {
	  MX_WARN(("mx_arch_memcpy_from_segment: sf_buf_alloc() failed !\n"));
	  status = ENOMEM;
	  goto exit;
	}
	vaddr = (char *)sf_buf_kva(sf);
	bcopy(ptr, vaddr+offset, to_copy);
	sf_buf_free(sf);
	remaining -= to_copy;
	ptr += to_copy;
	phys += to_copy;
	offset = 0;
      }
    }
    break;
  default:
    status = EINVAL;
    break;
  }
 exit:
  return status;
}

#endif /* _mx_arch_klib_h */
